%-------------------------------------------------------------------------%
%--------------------- Earth-to-Mars Orbit Transfer ----------------------%
%-------------------------------------------------------------------------%
clear all; clc

%--------------------------------------------------------------------------%
%----------------- Set Up Auxiliary Data for Problem ----------------------%
%--------------------------------------------------------------------------%
auxdata.mu = 1;
auxdata.T  = 0.1405;
auxdata.ve = 1.8758;
auxdata.use_angular_control = false;

%--------------------------------------------------------------------------%
%--------------- Set Up Bounds on State, Control, and Time ----------------%
%--------------------------------------------------------------------------%
t0          = 0;
tfmin       = 1;   tfmax       = 10;
r0          = 1;   rf          = 1.5;
theta0      = 0;
u0          = 0;   uf         = 0; 
v0          = sqrt(auxdata.mu/r0);  vf     = sqrt(auxdata.mu/rf);
rmin        = 1;   rmax        = 10; 
thetamin    = -pi; thetamax    = pi;
umin       = -10;  umax       = 10;
vmin       = -10;  vmax   = 10;
m0          = 1;
mmin        = 0.1; mmax        = m0;
alphamin    = -2*pi; alphamax = -alphamin;
w1min       = -1;  w1max       = 1;
w2min       = -1;  w2max       = 1;

bounds.phase.initialtime.lower  = t0;
bounds.phase.initialtime.upper  = t0;
bounds.phase.finaltime.lower    = tfmin;
bounds.phase.finaltime.upper    = tfmax;
bounds.phase.initialstate.lower = [r0, theta0, u0, v0, m0];
bounds.phase.initialstate.upper = [r0, theta0, u0, v0, m0];
bounds.phase.state.lower        = [rmin, thetamin, umin, vmin, mmin];
bounds.phase.state.upper        = [rmax, thetamax, umax, vmax, mmax];
bounds.phase.finalstate.lower   = [rf, thetamin, uf, vf, mmin];
bounds.phase.finalstate.upper   = [rf, thetamax, uf, vf, mmax];
if auxdata.use_angular_control,
    bounds.phase.control.lower      = [alphamin];    
    bounds.phase.control.upper      = [alphamax];    
else
    bounds.phase.control.lower      = [w1min, w2min];
    bounds.phase.control.upper      = [w1max, w2max];
    bounds.phase.path.lower         = 1;
    bounds.phase.path.upper         = 1;
end

%--------------------------------------------------------------------------%
%------------------------- Set Up Initial Guess ---------------------------%
%--------------------------------------------------------------------------%
tGuess              = [t0; tfmax];
rGuess              = [r0; rf];
thetaGuess          = [theta0; pi];
uGuess              = [u0; uf];
vGuess              = [v0; vf];
mGuess              = [m0; mmin];
alphaGuess          = [0; pi];    
w1Guess             = [0; -1];
w2Guess             = [1; 0];
guess.phase.time    = [tGuess];
guess.phase.state   = [rGuess, thetaGuess, uGuess, vGuess, mGuess];
if auxdata.use_angular_control,
    guess.phase.control = alphaGuess;
else
    guess.phase.control = [w1Guess, w2Guess];
end

%--------------------------------------------------------------------------%
%------------------------- Set Up Initial Mesh ----------------------------%
%--------------------------------------------------------------------------%
N = 10;
mesh.method = 'hp-LiuRao-Legendre';
mesh.phase.colpoints = 3*ones(1,N);
mesh.phase.fraction   = ones(1,N)/N;
mesh.tolerance = 1e-6;
mesh.maxiterations = 10;

%--------------------------------------------------------------------------%
%-------------------------- Set Up for Solver -----------------------------%
%--------------------------------------------------------------------------%
setup.name = 'Orbit-Raising-Problem';
setup.functions.continuous = @orbitRaisingContinuous;
setup.functions.endpoint = @orbitRaisingEndpoint;

setup.displaylevel = 2;
setup.nlp.solver = 'ipopt';
setup.auxdata = auxdata;
setup.bounds = bounds;
setup.guess = guess;
setup.derivatives.supplier = 'sparseCD';
setup.derivatives.derivativelevel = 'second';
setup.mesh = mesh;
setup.scales.method = 'automatic-hybrid';

%--------------------------------------------------------------------------%
%-------------------- Solve Problem and Extract Solution ------------------%
%--------------------------------------------------------------------------%
output = gpops2(setup);
