% ------------------------------------- %
%        Goddard Rocket Problem         %
%       (Three-Phase Formulation)       %
% ------------------------------------- %
% The code below implements the maximum %
% ascent of a one-dimensional sounding  %
% rocket.  The formulation in the code  %
% below divides the problem into three  %
% phases and enforces the singular arc  %
% conditions in the second phase. It is %
% noted that the problem formulation is %
% implemented using SI units.           %
% ------------------------------------- %
clear all
close all
clc

% ------------------------------------- %
%   Physical Constants Used in Problem  %
% ------------------------------------- %
g             = 9.80665;
sigma         = 2.629360578616578e-03;
H             = 7254.24;
c             = 481.8712825306375;

% ------------------------------------- %
%      Store Physical Constants in      %
%       Auxiliary Data Structure        %
% ------------------------------------- %
auxdata.g     = g;
auxdata.sigma = sigma;
auxdata.H     = H;
auxdata.c     = c;

t0 = 0;
tfMin = 0;    tfMax = 1000;
h0 = 0;       v0 = 0;
m0 = 43.7817; mf = 14.5939;
hMin = 0;     hMax = 30000;
vMin = -1500; vMax = +1500;
mMin = mf;    mMax = m0;
TMin = 0;     TMax = 858.702;
Deltat2Min = 1; Deltat2Max = 1000;

% ------------------------------------- %
%             Phase 1 Bounds            %
% ------------------------------------- %
p = 1;
bounds.phase(p).initialtime.lower  = t0;
bounds.phase(p).initialtime.upper  = t0;
bounds.phase(p).finaltime.lower    = tfMin;
bounds.phase(p).finaltime.upper    = tfMax;
bounds.phase(p).initialstate.lower = [h0,v0,m0]; 
bounds.phase(p).initialstate.upper = [h0,v0,m0]; 
bounds.phase(p).state.lower        = [hMin,vMin,mMin]; 
bounds.phase(p).state.upper        = [hMax,vMax,mMax]; 
bounds.phase(p).finalstate.lower   = [hMin,vMin,mMin]; 
bounds.phase(p).finalstate.upper   = [hMax,vMax,mMax]; 
bounds.phase(p).control.lower      = [TMax];
bounds.phase(p).control.upper      = [TMax];

% ------------------------------------- %
%             Phase 2 Bounds            %
% ------------------------------------- %
p = 2;
bounds.phase(p).initialtime.lower  = tfMin;
bounds.phase(p).initialtime.upper  = tfMax;
bounds.phase(p).finaltime.lower    = tfMin;
bounds.phase(p).finaltime.upper    = tfMax;
bounds.phase(p).initialstate.lower = [hMin,vMin,mMin]; 
bounds.phase(p).initialstate.upper = [hMax,vMax,mMax]; 
bounds.phase(p).state.lower        = [hMin,vMin,mMin]; 
bounds.phase(p).state.upper        = [hMax,vMax,mMax]; 
bounds.phase(p).finalstate.lower   = [hMin,vMin,mMin]; 
bounds.phase(p).finalstate.upper   = [hMax,vMax,mMax]; 
bounds.phase(p).control.lower      = [TMin];
bounds.phase(p).control.upper      = [TMax];
bounds.phase(p).path.lower         = 0;
bounds.phase(p).path.upper         = 0;
bounds.phase(p).duration.lower     = [Deltat2Min];
bounds.phase(p).duration.upper     = [Deltat2Max];

% ------------------------------------- %
%             Phase 3 Bounds            %
% ------------------------------------- %
p = 3;
bounds.phase(p).initialtime.lower  = tfMin;
bounds.phase(p).initialtime.upper  = tfMax;
bounds.phase(p).finaltime.lower    = tfMin;
bounds.phase(p).finaltime.upper    = tfMax;
bounds.phase(p).initialstate.lower = [hMin,vMin,mMin]; 
bounds.phase(p).initialstate.upper = [hMax,vMax,mMax]; 
bounds.phase(p).state.lower        = [hMin,vMin,mMin]; 
bounds.phase(p).state.upper        = [hMax,vMax,mMax]; 
bounds.phase(p).finalstate.lower   = [hMin,vMin,mf]; 
bounds.phase(p).finalstate.upper   = [hMax,vMax,mf]; 
bounds.phase(p).control.lower      = [TMin];
bounds.phase(p).control.upper      = [TMin];

% ------------------------------------- %
%            Event Group 1:             %
%     Linkage Constraint on Time and    %
%   State Between Phase 1 and Phase 2   %
% ------------------------------------- %
bounds.eventgroup(1).lower = [zeros(1,4)];
bounds.eventgroup(1).upper = [zeros(1,4)];

% ------------------------------------- %
%            Event Group 2:             %
%     Linkage Constraint on Time and    %
%   State Between Phase 2 and Phase 3   %
% ------------------------------------- %
bounds.eventgroup(2).lower = [zeros(1,4)];
bounds.eventgroup(2).upper = [zeros(1,4)];

% ------------------------------------- %
%            Event Group 3:             %
%       Condition That Enforces         %
%        Start of Singular Arc          %
% ------------------------------------- %
bounds.eventgroup(3).lower = [0];
bounds.eventgroup(3).upper = [0];

% ------------------------------------- %
%    Guess of Solution in Each Phase    %
% ------------------------------------- %
tGuess{1} = [t0; 50];
tGuess{2} = [50; 100];
tGuess{3} = [100; 150];
hGuess{1} = [h0; h0];
hGuess{2} = [h0; h0];
hGuess{3} = [h0; h0];
vGuess{1} = [v0; v0];
vGuess{2} = [50; 50];
vGuess{3} = [v0; v0];
mGuess{1} = [m0; mf];
mGuess{2} = [m0; mf];
mGuess{3} = [m0; mf];
TGuess{1} = [TMax; TMax];
TGuess{2} = [TMax; TMin];
TGuess{3} = [TMax; TMin];
for p=1:3,
    guess.phase(p).time    = tGuess{p};
    guess.phase(p).state   = [hGuess{p},vGuess{p},mGuess{p}];
    guess.phase(p).control = TGuess{p};
end

% ------------------------------------- %
%     Specification of Initial Mesh     %
% ------------------------------------- %
mesh.method             = 'hp-LiuRao-Legendre';
mesh.tolerance          = 1e-7;
mesh.maxiterations      = 5;
mesh.colpointsmin       = 3;
mesh.colpointsmax       = 14;
for p=1:3,
    mesh.phase(p).colpoints = 3*ones(1,10);
    mesh.phase(p).fraction  = 0.1*ones(1,10);
end

% ------------------------------------- %
%      Setup of Problem for GPOPS-II    %
% ------------------------------------- %
setup.name                           = 'Goddard-Rocket-Problem';
setup.auxdata                        = auxdata;
setup.functions.continuous           = @goddardRocketContinuous;
setup.functions.endpoint             = @goddardRocketEndpoint;
setup.displaylevel                   = 2;
setup.bounds                         = bounds;
setup.guess                          = guess;
setup.mesh                           = mesh;
setup.nlp.solver                     = 'ipopt';
setup.nlp.ipoptoptions.linear_solver = 'ma57';
setup.nlp.ipoptoptions.tolerance     = 1e-7;
setup.derivatives.supplier           = 'sparseCD';
setup.derivatives.derivativelevel    = 'second';
setup.method                         = 'RPM-Differentiation';

% ------------------------------------- %
%      Solve Problem Using GPOPS-II     %
% ------------------------------------- %
output = gpops2(setup);

