/*
 * mod_authz_svndb.c: an Apache mod_dav_svn sub-module to provide path
 *                    based authorization for a Subversion repository.
 *
 * ====================================================================
 * Copyright (c) 2003-2005 CollabNet.  All rights reserved.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at http://subversion.tigris.org/license-1.html.
 * If newer versions of this license are posted there, you may use a
 * newer version instead, at your option.
 *
 * This software consists of voluntary contributions made by many
 * individuals.  For exact contribution history, see the revision
 * history and logs, available at http://subversion.tigris.org/.
 * ====================================================================
 */

#include <httpd.h>
#include <http_config.h>
#include <http_core.h>
#include <http_request.h>
#include <http_protocol.h>
#include <http_log.h>

#include <ap_config.h>
#include <apr_uri.h>
#include <ctype.h>

#include "special.h"
#include "mod_authz_svndb.h"

#include "apr_dbd.h"
#include "mod_dbd.h"
#include "apr_strings.h"

extern module AP_MODULE_DECLARE_DATA authz_svndb_module;
static ap_dbd_t *(*dbd_open)(apr_pool_t*, server_rec*) = NULL;
static void (*dbd_close)(server_rec*, ap_dbd_t*) = NULL;
static const char *const noerror = "???";

/* * * Configuration * * */

/* This is called once for every <Location> directive to 
 * instantiate the configuration structure - cfg */
static void *authz_svndb_init(apr_pool_t *p, char *d) {
   char *tmp;
   authz_svndb_cfg *cfg = apr_pcalloc(p, sizeof(authz_svndb_cfg));

   if (d) {
      tmp = apr_pstrcat(p, d, "/", NULL);
      ap_getparents(tmp);
      ap_no2slash(tmp);
      cfg->uri_base_path = tmp;
   } else {
      cfg->uri_base_path = NULL;
   }

   /* defaults */
   cfg->authoritative = 2;
   cfg->ra = authz_svndb_none;
   cfg->special_prefix = NULL;
   cfg->pt = authz_svndb_unknown_path;

   if (dbd_open == NULL) {
      dbd_open = APR_RETRIEVE_OPTIONAL_FN(ap_dbd_open);
   }

   if (dbd_close == NULL) {
      dbd_close = APR_RETRIEVE_OPTIONAL_FN(ap_dbd_close);
   }

   return cfg;
}

/* This is called once for every <Location> directive to merge
 * settings with any that may be defined higher up in the url
 * hierarchy */
static void *authz_svndb_merge(apr_pool_t *pool, void *BASE, void *ADD) {
   authz_svndb_cfg *base = BASE;
   authz_svndb_cfg *add = ADD;
   authz_svndb_cfg *cfg = apr_palloc(pool, sizeof(authz_svndb_cfg));

   cfg->pt =
      (base->pt) ?
         base->pt : add->pt;

   cfg->uri_base_path =
      (base->uri_base_path) ?
         base->uri_base_path : add->uri_base_path;

   cfg->special_prefix =
      (add->special_prefix) ?
         add->special_prefix : base->special_prefix;

   cfg->authoritative =
      (add->authoritative < 2) ?
         add->authoritative : base->authoritative;

   if (add->ra == authz_svndb_none) {
      cfg->ra = add->ra;
   } else {
      cfg->ra = base->ra | add->ra;
      if (add->ra & authz_svndb_read) {
         cfg->ra = (cfg->ra & ~authz_svndb_deny_read);
      }
      if (add->ra & authz_svndb_deny_read) {
         cfg->ra = (cfg->ra & ~authz_svndb_read);
      }
      if (add->ra & authz_svndb_write) {
         cfg->ra = (cfg->ra & ~authz_svndb_deny_write);
      }
      if (add->ra & authz_svndb_deny_write) {
         cfg->ra = (cfg->ra & ~authz_svndb_write);
      }
   }

   return cfg;
}

/* This is triggered by the AuthzSVNDBAccess configuration flag */
static const char *set_required_access(
   cmd_parms *cmd,
   void *cfg,
   const char *required_access
) {
   char *error = NULL;
   authz_svndb_cfg *mycfg = (authz_svndb_cfg*)cfg;

   if (!strcasecmp(required_access, "none")) {
      mycfg->ra = authz_svndb_none;
   } else if (!strcasecmp(required_access, "+r")) {
      mycfg->ra = authz_svndb_read;
   } else if (!strcasecmp(required_access, "+w")) {
      mycfg->ra = authz_svndb_write;
   } else if (!strcasecmp(required_access, "+rw")) {
      mycfg->ra = authz_svndb_read | authz_svndb_write;
   } else if (!strcasecmp(required_access, "-r")) {
      mycfg->ra = authz_svndb_deny_read;
   } else if (!strcasecmp(required_access, "-w")) {
      mycfg->ra = authz_svndb_deny_write;
   } else if (!strcasecmp(required_access, "-rw")) {
      mycfg->ra = authz_svndb_deny_read | authz_svndb_deny_write;
   } else {
      error = apr_psprintf(
         cmd->pool,
         "Invalid value for directive AuthzSVNDBAccess, "
         "expected one of 'none', '+r', '+w', '+rw', '-r', "
         "'-w', or '-rw'."
      );
   }

   return error;
}

/* This is triggered by the SVNSpecialURI configuration flag */
static const char *set_special_prefix(
   cmd_parms *cmd,
   void *cfg,
   const char *path
) {
   apr_size_t length;
   const char *error = NULL;
   char *tmp = apr_pstrdup(cmd->pool, path);

   /* apply a bit of processing to the uri:
    * - eliminate .. and . components
    * - eliminate double slashes
    * - eliminate leading and trailing slashes
    */
   ap_getparents(tmp);
   ap_no2slash(tmp);

   if (*tmp == '/') tmp++;

   length = strlen(tmp);
   if (length == 0) {
      error = "The special URI path must have at least one component.";
   } else if (tmp[length - 1] == '/') {
      tmp[length - 1] = '\0';
   }

   if (!error) {
      ((authz_svndb_cfg*)cfg)->special_prefix = tmp;
   }

   return error;
}

/* This is triggered by the SVNParentPath and SVNPath configuration
 * flags.  cmd->info will contain and enum of type path_type which
 * indicates wether the uri_base_path from the root <Location>
 * directive contains the repository name or the repository name
 * is to be matched from the next component in the url.  We do not
 * use the filesystem location (path) in this module, so its value
 * is discarded. */
static const char *set_path_type(
   cmd_parms *cmd,
   void *cfg,
   const char *path
) {
   authz_svndb_cfg *mycfg = (authz_svndb_cfg*)cfg;

   if (path) {
      if (!mycfg->pt) {
         mycfg->pt = (path_type)cmd->info;
      }
   }

   return NULL;
}

static const command_rec authz_svndb_cmds[] = {
   AP_INIT_FLAG(
      "AuthzSVNDBAuthoritative", ap_set_flag_slot,
      (void*)APR_OFFSETOF(authz_svndb_cfg, authoritative), ACCESS_CONF,
      "Set to 'off' to allow access control to be passed along to "
      "lower modules. (default = on)"
   ),
   AP_INIT_TAKE1(
      "AuthzSVNDBAccess", set_required_access, NULL, ACCESS_CONF,
      "Set to one of 'none', '+r', '+w', '+rw', '-r', '-w', or '-rw'. "
      "(default = +r)"
   ),

   /* Grab some of mod_dav_svn's settings for our own use. */

   AP_INIT_TAKE1(
      "SVNSpecialURI", set_special_prefix, NULL, RSRC_CONF,
      "Specify the URI component for special subversion resources. "
      "(default = !svn)"
   ),

   AP_INIT_TAKE1(
      "SVNPath", set_path_type,
      (void*)authz_svndb_direct_path, ACCESS_CONF,
      "Specifies the location in the filesystem for a Subversion "
      "repository's files. (default is null)"
   ),

   AP_INIT_TAKE1(
      "SVNParentPath", set_path_type,
      (void*)authz_svndb_parent_path, ACCESS_CONF,
      "Specifies the location in the filesystem whose subdirectories are "
      "assumed to be Subversion repositories. (default is null)"
   ),

   { NULL }
};

/* * * Access Control * * */

/* Merges the required_access from the AuthzSVNDBAccess flag with the
 * access requirements based on the requested method to determine the
 * final access requirements of the request. */
static required_access_type get_required_access(
   int requested_method,
   required_access_type required_access
) {
   switch (requested_method) {
      /* All methods requiring read access to all subtrees of r->uri */
      case M_COPY:
         required_access |= authz_svndb_recursive;

      /* All methods requiring read access to r->uri */
      case M_OPTIONS:
      case M_GET:
      case M_PROPFIND:
      case M_REPORT:
         required_access |= authz_svndb_read;
         break;

      /* All methods requiring write access to all subtrees of r->uri */
      case M_MOVE:
      case M_DELETE:
         required_access |= authz_svndb_recursive;

      /* All methods requiring write access to r->uri */
      case M_MKCOL:
      case M_PUT:
      case M_PROPPATCH:
      case M_CHECKOUT:
      case M_MERGE:
      case M_MKACTIVITY:
      case M_LOCK:
      case M_UNLOCK:
         required_access |= authz_svndb_write;
         break;

      /* Require most strict access for unknown methods */
      default:
         required_access |=
            authz_svndb_read | authz_svndb_write | authz_svndb_recursive;
         break;
   }

   return required_access;
}

/* Returns the repos_name & repos_path based on the requested uri. */
static int split_uri(
   request_rec *r,
   authz_svndb_cfg *cfg,
   const char *uri,
   const char **repos_name,
   const char **repos_path
) {
   int status = OK;
   int offset, components, bp_length, pp_length, pf_length, hr_length;
   const struct special_path *path;
   const char *tmp, *private_path, *prefix =
      cfg->special_prefix ?
         cfg->special_prefix : SVN_DEFAULT_SPECIAL_PREFIX;
   
   if (!cfg->pt) {
      /* SVN[Parent]Path not specified. */
      status = HTTP_INTERNAL_SERVER_ERROR;
   } else {
      tmp = apr_pstrdup(r->pool, uri);
      ap_getparents((char*)tmp);
      ap_no2slash((char*)tmp);
      /* uri_base_path is always from a <Location> directive so it always ends
       * in a '/' (this is guaranteed by authz_svndb_init())
       * uri may or may not end in a slash depending on what is being
       * requested.
       */
      private_path = ap_stripprefix(tmp, cfg->uri_base_path);
      private_path--; /* re-prepend the '/' that uri_base_path matched */

      if (cfg->pt == authz_svndb_direct_path) {
         offset = bp_length = strlen(cfg->uri_base_path);
         if (bp_length <= 1) {
            /* No repository name specified. */
            status = HTTP_FORBIDDEN;
         } else {
            while(--offset > 0 && cfg->uri_base_path[offset - 1] != '/');
            *repos_name = apr_pstrmemdup(
               r->pool, cfg->uri_base_path + offset, bp_length - offset - 1
            );
         }
      } else if (cfg->pt == authz_svndb_parent_path) {
         if (private_path[1] == '\0') {
            /* No repository name specified. */
            status = HTTP_FORBIDDEN;
         } else {
            tmp = ap_strchr_c(private_path + 1, '/');
            if (!tmp) {
               *repos_name = private_path + 1;
               private_path = "/";
            } else {
               *repos_name = apr_pstrndup(
                  r->pool, private_path + 1, tmp - private_path - 1
               );
               private_path = tmp;
            }
         }
      }

      if (status == HTTP_FORBIDDEN) {
         ap_log_rerror(
            APLOG_MARK,
            APLOG_ERR,
            0,
            r,
            "authz_svndb: split_uri() - Missing Repository Name"
         );
      }

      if (status == OK) {
         pp_length = strlen(private_path);
         pf_length = strlen(prefix);

         if (
            pp_length > pf_length &&
            private_path[pf_length + 1] == '/' &&
            memcmp(
               private_path + 1,
               prefix,
               pf_length
            ) == 0
         ) {
            private_path += pf_length + 1;
            pp_length -= pf_length + 1;

            path = special_paths;
            while(path->header != NULL) {
               hr_length = strlen(path->header);

               if (
                  pp_length > hr_length && 
                  (
                     private_path[hr_length + 1] == '/' ||
                     (
                        private_path[hr_length + 1] == '\0' &&
                        path->components == 0
                     )
                  ) &&
                  memcmp(
                     private_path + 1,
                     path->header,
                     hr_length
                  ) == 0
               ) {
                  components = path->components;
                  tmp = private_path + hr_length + 1;
                  while (components && tmp) {
                     tmp = ap_strchr_c(tmp + 1, '/');
                     components--;
                  }

                  if (tmp && tmp[0]) {
                     private_path = tmp;
                  } else {
                     if (components == 0) {
                        if (!path->private_path) {
                           private_path = NULL;
                        } else {
                           private_path = "/";
                        }
                     } else {
                        /* Not Enough Components */
                        status = HTTP_INTERNAL_SERVER_ERROR;
                        ap_log_rerror(
                           APLOG_MARK,
                           APLOG_ERR,
                           0,
                           r,
                           "authz_svndb: split_uri() - Invalid /!svn/%s/..",
                           path->header
                        );
                     }
                  }

                  break;
               }

               path++;
            }

            if (path->header == NULL) {
               /* Header Unknown or Missing Special Components */
               status = HTTP_INTERNAL_SERVER_ERROR;
               ap_log_rerror(
                  APLOG_MARK,
                  APLOG_ERR,
                  0,
                  r,
                  "authz_svndb: split_uri() - Invalid /!svn/.."
               );
            }
         }

         if (status == OK) {
            *repos_path = apr_pstrdup(r->pool, private_path);
         }
      }
   }

   return status;
}

/* Returns 1 (true) if all regex patterns (repository (r), path (p), and
 * entity (e)) match the information from the request (repos_name, repos_path,
 * and user). Otherwise, returns 0 (false). */
static int eval_regex(
   apr_pool_t *pool,
   const char *r, const char *repos_name,
   const char *p, const char *repos_path,
   const char *e, const char *user,
   apr_array_header_t *groups,
   const char **message
) {
   int n;
   const char *i, *mb, *me;
   char *o, *b;
   ap_regex_t *regex;
   ap_regmatch_t regmatch[AP_MAX_REG_MATCH];

   if (*r != '\0') {
      regex = ap_pregcomp(
         pool, r, AP_REG_EXTENDED
      );

      if (ap_regexec(
         regex,
         repos_name,
         AP_MAX_REG_MATCH,
         regmatch,
         0
      )) {
         return 0;
      }
   }

   if (repos_path && *p != '\0') {
      regex = ap_pregcomp(
         pool, p, AP_REG_EXTENDED
      );

      if (ap_regexec(
         regex,
         repos_path,
         AP_MAX_REG_MATCH,
         regmatch,
         0
      )) {
         return 0;
      }
   }

   if (*e != '\0') {
      i = e;
      o = b = apr_palloc(pool, PATH_MAX);
      while (o-b < PATH_MAX-1) {
         if (*i == '\0') break;
         if (*i != '$' || !isdigit(*(i+1))) {
            *o++ = *i++;
            continue;
         }
         /* If the entity contains a '$' and
          * we are doing a global access lookup
          * we assume that the entity would
          * match */
         if (!repos_path) return 1;
         n = strtol((i+1), (char**)&i, 10);
         if (
            n < 0 ||
            n >= AP_MAX_REG_MATCH ||
            regmatch[n].rm_eo <= regmatch[n].rm_so
         ) {
            *message = apr_psprintf(
               pool,
               "Invalid Reference $%d",
               n
            );
            return 0;
         }
         mb = repos_path + regmatch[n].rm_so;
         me = repos_path + regmatch[n].rm_eo;
         while (o-b < PATH_MAX-1 && mb < me) {
            *o++ = *mb++;
         }
      }
      *o = '\0';

      if (*e != '@') {
         /* We know that user is non-null
          * because *e != '\0' */
         if (strcmp(user, b)) {
            return 0;
         }
      } else {
         for (n=0; n < groups->nelts; n++) {
            if (!strcmp(
               (b+1), ((const char**)groups->elts)[n]
            )) {
               break;
            }
         }
         if (n == groups->nelts) return 0;
      }
   }

   return 1;
}

/* Returns status (OK, DECLINED, or HTTP_FORBIDDEN) depending on wether a
 * matching rule was found and wether it granted or denied access to the 
 * request.  May return HTTP_INTERNAL_SERVER_ERROR if an error accured. */
static int svndb_get_authorization(
   request_rec *rq,
   const char **repos_name,
   const char **repos_path,
   required_access_type required_access
) {
   int status = DECLINED;
   int n, tmp, vector = -1, rv = 0;
   int required_access_level = (required_access & 0x2) ? 2 : 1;
   apr_dbd_results_t *res = NULL;
   apr_dbd_row_t *row = NULL;
   apr_array_header_t *groups = NULL;
   ap_dbd_t *dbd = NULL;
   const char *i, *x, *r, *p, *e, *a, **group, *message = NULL;
   char *entity, *entity_regex, *query = NULL;
   char *query_pgsql_recursive = NULL;

   if (dbd_open != NULL) {
      dbd = dbd_open(rq->pool, rq->server);
      if (dbd) {
         if (rq->user) {
            /* Get groups to which user belongs. */
            if ((rv = apr_dbd_select(
               dbd->driver,
               rq->pool,
               dbd->handle,
               &res,
               apr_psprintf(rq->pool,
                  "select svngroup from svngroups where svnmember='%s'",
               rq->user),
               1
            )) == 0) {
               n = apr_dbd_num_tuples(dbd->driver, res);
               groups = apr_array_make(rq->pool, n, sizeof(const char*));
               while ((rv = apr_dbd_get_row(
                  dbd->driver, rq->pool, res, &row, -1
               )) != -1) {
                  if (rv == 0) {
                     group = apr_array_push(groups);
                     *group = apr_pstrdup(
                        rq->pool, apr_dbd_get_entry(dbd->driver, row, 0)
                     );
                  } else {
                     break;
                  }
               }
            }
         }

         if (!(rv == 0 || rv == -1)) {
            message = apr_dbd_error(dbd->driver, dbd->handle, rv);
         } else {
            /* Define Queries. */
            x = apr_dbd_name(dbd->driver);
            if (!strcmp(x, "pgsql")) {
               entity = "entity is null";

               if (rq->user) {
                  entity = apr_pstrcat(
                     rq->pool, entity, " or entity in ('", rq->user, "'", NULL
                  );

                  for (n=0; n < groups->nelts; n++) {
                     entity = apr_pstrcat(
                        rq->pool,
                        entity,
                        ", '@",
                        ((const char**)groups->elts)[n],
                        "'",
                        NULL
                     );
                  }

                  entity = apr_pstrcat(rq->pool, entity, ")", NULL);
                  entity_regex = apr_pstrcat(
                     rq->pool,
                     "entity like '%%$%%' or ",
                     entity,
                     NULL
                  );
               } else {
                  entity_regex = entity;
               }

               if (*repos_path) {
                  query = apr_psprintf(
                     rq->pool,
                     "select regex, repository, path, entity, chr(access + 48) "
                     "from svnauthz where ("
                        "("
                           "regex='t' and (%s)"
                        ") or ("
                           "regex='f' and (%s) "
                           "and (repository is null or repository='%s') "
                           "and ("
                              "path is null "
                              "or (path='/' and '%s'='/') "
                              "or rtrim(path, '/')=rtrim('%s', '/') "
                              "or ("
                                 "substr(path, length(path), 1)='/' "
                                 "and path=substr('%s', 1, length(path))"
                              ")"
                           ")"
                        ")"
                     ") order by "
                        "regex, "
                        "length(path), "
                        "substr(entity, 1, 1) desc, "
                        "access",
                     entity_regex,
                     entity,
                     *repos_name,
                     *repos_path,
                     *repos_path,
                     *repos_path
                  );

                  query_pgsql_recursive = apr_psprintf(
                     rq->pool,
                     "select regex, repository, path "
                     "from svnauthz "
                     "where access < %d "
                     "and ("
                        "("
                           "regex='t' and (%s)"
                        ") or ("
                           "regex='f' "
                           "and (%s) "
                           "and ("
                              "repository is null "
                              "or repository='%s'"
                           ")"
                           "and ("
                              "path like rtrim('%s', '/') || '/%%'"
                           ")"
                        ")"
                     ") order by regex",
                     required_access_level,
                     entity_regex,
                     entity,
                     *repos_name,
                     *repos_path
                  );
               } else {
                  /* Perform a "global access lookup". */
                  query = apr_psprintf(
                     rq->pool,
                     "select regex, repository, path, entity "
                     "from svnauthz where access >= %d and ("
                        "("
                           "regex='t' and (%s)"
                        ") or ("
                           "regex='f' "
                           "and (%s) "
                           "and ("
                              "repository is null "
                              "or repository='%s'"
                           ")"
                        ")"
                     ") order by regex",
                     required_access_level,
                     entity_regex,
                     entity,
                     *repos_name
                  );
               }
            } else {
               message = apr_psprintf(
                  rq->pool, "Sorry, %s is not yet supported.", x
               );
            }
         }

         /* Perform actual database lookup. */
         if (!message) {
            if ((rv = apr_dbd_select(
               dbd->driver,
               rq->pool,
               dbd->handle,
               &res,
               query,
               1
            )) == 0) {
               if (!*repos_path) {
                  /* Path is null, so do a global access lookup. */
                  if ((rv = apr_dbd_get_row(
                     dbd->driver, rq->pool, res, &row, -1
                  )) != -1) {
                     if (rv == 0) {
                        x = apr_dbd_get_entry(dbd->driver, row, 0);
                        if (*x == 't') {
                           do {
                              if (rv == 0) {
                                 r = apr_dbd_get_entry(dbd->driver, row, 1);
                                 p = apr_dbd_get_entry(dbd->driver, row, 2);
                                 e = apr_dbd_get_entry(dbd->driver, row, 3);

                                 if (eval_regex(
                                    rq->pool,
                                    r, *repos_name,
                                    p, *repos_path,
                                    e, rq->user, groups,
                                    &message
                                 )) {
                                    status = OK;
                                    break;
                                 }
                              } else {
                                 break;
                              }
                           } while ((rv = apr_dbd_get_row(
                              dbd->driver, rq->pool, res, &row, -1
                           )) != -1);
                        } else {
                           status = OK;
                        }
                     }
                  }
               } else {
                  /* Figure out which results apply and which one wins. */
                  while ((rv = apr_dbd_get_row(
                     dbd->driver, rq->pool, res, &row, -1
                  )) != -1) {
                     if (rv == 0) {
                        x = apr_dbd_get_entry(dbd->driver, row, 0);
                        r = apr_dbd_get_entry(dbd->driver, row, 1);
                        p = apr_dbd_get_entry(dbd->driver, row, 2);
                        e = apr_dbd_get_entry(dbd->driver, row, 3);
                        a = apr_dbd_get_entry(dbd->driver, row, 4);

                        /* Is Access being Granted or Denied: */
                        tmp = ((*a - 48) >= required_access_level) ? 0 : 1;

                        /* To Whom: */
                        switch (*e) {
                           case '\0': break;
                           case '@': tmp |= 2; break;
                           default: tmp |= 4; break;
                        }

                        if (*x == 't') {
                           if (eval_regex(
                              rq->pool,
                              r, *repos_name,
                              p, *repos_path,
                              e, rq->user, groups,
                              &message
                           )) {
                              p = *repos_path;
                           } else {
                              if (message) {
                                 vector = -1;
                                 break;
                              } else {
                                 continue;
                              }
                           }
                        }

                        /* At what Path Depth: */
                        for (i = p; *i != '\0'; i++);
                        /* Invert the sense of which is a deeper Path -
                         * One with a trailing slash or One without it */
                        if (*x == 'f') {
                           if ((i-p)>0) {
                              if (*(i-1)=='/') {
                                 i--;
                              } else {
                                 i++;
                              }
                           } else {
                              i++;
                           }
                        }
                        tmp |= ((i-p) << 3);

                        /* Was a repository specified: */
                        tmp |= ((*r == '\0') ? 0 : 1) << 30;

                        if (tmp > vector) vector = tmp;
                     } else {
                        break;
                     }
                  }

                  if (rv == 0 || rv == -1) {
                     if (vector >= 0) {
                        if (rq->user) {
                           status = (vector & 1) ? HTTP_FORBIDDEN : OK;
                        } else {
                           if (!(vector & 1)) status = OK;
                        }
                     }
                  } else {
                     message = apr_dbd_error(dbd->driver, dbd->handle, rv);
                  }

                  /* Check for recursive access if necessary. */
                  if (status == OK && (required_access & authz_svndb_recursive)) {
                     if ((rv = apr_dbd_select(
                        dbd->driver,
                        rq->pool,
                        dbd->handle,
                        &res,
                        query_pgsql_recursive,
                        1
                     )) == 0) {
                        while ((rv = apr_dbd_get_row(
                           dbd->driver, rq->pool, res, &row, -1
                        )) != -1) {
                           if (rv == 0) {
                              x = apr_dbd_get_entry(dbd->driver, row, 0);
                              if (*x == 'f') {
                                 status = HTTP_FORBIDDEN;
                                 break;
                              } else {
                                 r = apr_dbd_get_entry(dbd->driver, row, 1);
                                 p = apr_dbd_get_entry(dbd->driver, row, 2);
                                 e = apr_dbd_get_entry(dbd->driver, row, 3);

                                 if (eval_regex(
                                    rq->pool,
                                    r, *repos_name,
                                    p, *repos_path,
                                    e, rq->user, groups,
                                    &message
                                 )) {
                                    status = HTTP_FORBIDDEN;
                                    break;
                                 } else {
                                    if (message) {
                                       status = DECLINED;
                                       break;
                                    }
                                 }
                              }
                           }
                        }
                     }
                  }
               }
            }

            if (!(rv == 0 || rv == -1)) {
               message = apr_dbd_error(dbd->driver, dbd->handle, rv);
            }
         }

         if (dbd_close != NULL) dbd_close(rq->server, dbd);
      }
   } else {
      message = "You must load mod_dbd to enable AuthzSVNDB functions.";
   }

   if (message) {
      status = HTTP_INTERNAL_SERVER_ERROR;
      ap_log_rerror(
         APLOG_MARK,
         APLOG_ERR,
         0,
         rq,
         "authz_svndb: get_authorization() - repository='%s' user='%s' [%s]",
         apr_pstrcat(
            rq->pool, *repos_name, ":", *repos_path, NULL
         ),
         (rq->user) ? rq->user : "",
         (message) ? message : noerror
      );
   }

   return status;
}

/* Called by any one of the three hooks access_checker, check_user_id, or auth_checker */
/* Gets authorization for requested uri and destination uri (if present) */
static int svndb_check_access(
   request_rec *r,
   authz_svndb_cfg *cfg,
   const char **repos_path_ref,
   const char **dest_repos_path_ref
) {
   int status = OK;
   const char *repos_name = apr_table_get(r->notes, "authz_svndb-rn");
   const char *dest_repos_name = apr_table_get(r->notes, "authz_svndb-drn");
   const char *repos_path = apr_table_get(r->notes, "authz_svndb-rp");
   const char *dest_repos_path = apr_table_get(r->notes, "authz_svndb-drp");
   const char *dest_uri;
   apr_uri_t parsed_dest_uri;
   required_access_type ra = get_required_access(
      r->method_number,
      cfg->ra
   );

   if (
      ((ra & authz_svndb_read) && (ra & authz_svndb_deny_read)) ||
      ((ra & authz_svndb_write) && (ra & authz_svndb_deny_write))
   ) {
      status = HTTP_FORBIDDEN;
   } else if (!repos_name) {
      status = split_uri(
         r,
         cfg,
         r->uri,
         &repos_name,
         &repos_path
      );

      if (status == OK) {
         /* This function may get called multiple times per request (e.g.
          * when anonymous check failes and then a check on an authenticated
          * user is tried) so let's make a note the first time through to
          * save work in the future. */
         apr_table_setn(r->notes, "authz_svndb-rn", repos_name);
         if (repos_path) {
            apr_table_setn(r->notes, "authz_svndb-rp", repos_path);
         }
      }
   }

   if (status == OK) {
      *repos_path_ref = apr_pstrcat(
         r->pool, repos_name, ":", repos_path, NULL
      );

      /* Ignore the URI passed to MERGE, like mod_dav_svn does.
       * See issue #1821.
       * XXX: When we start accepting a broader range of DeltaV MERGE
       * XXX: requests, this should be revisited.
       */
      if (r->method_number == M_MERGE) {
          repos_path = NULL;
      }

      /*
       * first test the special case where repos_path == NULL, and skip
       * calling the authz routines in that case.  This is an oddity of
       * the DAV RA method: some requests have no repos_path, but apache
       * still triggers an authz lookup for the URI.
       *
       * However, if repos_path == NULL and the request requires write
       * access, then perform a global authz lookup.  The request is
       * denied if the user commiting isn't granted any access anywhere
       * in the repository.  This is to avoid operations that involve no
       * paths (commiting an empty revision, leaving a dangling
       * transaction in the FS) being granted by default, letting
       * unauthenticated users write some changes to the repository.
       * This was issue #2388.
       *
       * XXX: For now, requesting access to the entire repository always
       * XXX: succeeds, until we come up with a good way of figuring
       * XXX: this out.
       */
      if (repos_path || (ra & authz_svndb_write)) {
         /* XXX: MKCOL, MOVE, DELETE
          * XXX: Require write access to the parent dir of repos_path.
          */

         /* XXX: PUT
          * XXX: If the path doesn't exist, require write access to the
          * XXX: parent dir of repos_path.
          */
         if((status = svndb_get_authorization(
            r,
            &repos_name,
            &repos_path,
            ra
         )) == OK) {
            /* Only MOVE and COPY have a second uri to authorize. */
            if (r->method_number == M_MOVE || r->method_number == M_COPY) {
               if (!dest_repos_name) {
                  dest_uri = apr_table_get(r->headers_in, "Destination");

                  if (dest_uri) {
                     apr_uri_parse(r->pool, dest_uri, &parsed_dest_uri);
                     ap_unescape_url(parsed_dest_uri.path);
                     dest_uri = parsed_dest_uri.path;

                     /* If it is not the same location, then we don't allow it.
                      * XXX: Instead we could compare repository uuids, but
                      * XXX: that seems a bit over the top.
                      */
                     if (!strncmp(
                        dest_uri,
                        cfg->uri_base_path,
                        strlen(cfg->uri_base_path)
                     )) {
                        status = split_uri(
                           r,
                           cfg,
                           dest_uri,
                           &dest_repos_name,
                           &dest_repos_path
                        );

                        if (status == OK) {
                           apr_table_setn(
                              r->notes, "authz_svndb-drn", dest_repos_name
                           );
                           apr_table_setn(
                              r->notes, "authz_svndb-drp", dest_repos_path
                           );
                        }
                     } else {
                        status = HTTP_BAD_REQUEST;
                     }
                  } else {
                     /* Decline when there is no Destination uri. */
                     status = DECLINED;
                  }
               }

               if (status == OK) {
                  *dest_repos_path_ref = apr_pstrcat(
                     r->pool, dest_repos_name, ":", dest_repos_path, NULL
                  );

                  /* Check access on the destination repos_path.
                   * Again, skip this if repos_path == NULL
                   * (see above for explanations)
                   *
                   * XXX: MOVE and COPY, if the path doesn't exist yet,
                   * XXX: require write access to the parent dir of
                   * XXX: dest_repos_path.
                   */
                  if (dest_repos_path) {
                     status = svndb_get_authorization(
                        r,
                        &dest_repos_name,
                        &dest_repos_path,
                        authz_svndb_write | authz_svndb_recursive
                     );
                  }
               }
            }
         }
      }
   }

   return status;
}

/* Log a message indicating the access control decision made about a
 * request.  FILE and LINE should be supplied via the APLOG_MARK macro.
 * ALLOWED is a boolean value.  REPOS_PATH and DEST_REPOS_PATH are
 * information about the request.  DEST_REPOS_PATH may be NULL. */
static void log_access_verdict(
   const char *file, int line,
   const request_rec *r,
   int allowed,
   const char *repos_path,
   const char *dest_repos_path
) {
   char *message = apr_psprintf(
      r->pool,
      "Access %s: %s %s %s",
      (allowed) ? "granted" : "denied", /* verdict */
      (r->user) ? apr_psprintf(r->pool, "'%s'", r->user) : "-",
      r->method,
      repos_path
   );

   if (dest_repos_path) {
      message = apr_pstrcat(r->pool, message, " ", dest_repos_path);
   }

   ap_log_rerror(
      file, line,
      (allowed) ? APLOG_INFO : APLOG_ERR, /* log level */
      0,
      r, /* request */
      message
   );
}

/* * * Hooks * * */

/* Check Anonymous Access */
static int access_checker(request_rec *r) {
   authz_svndb_cfg *cfg = ap_get_module_config(
      r->per_dir_config,
      &authz_svndb_module
   );

   int status;
   const char *repos_path;
   const char *dest_repos_path = NULL;

   if (cfg->ra == authz_svndb_none) { /* access restrictions disabled */
      status = OK;
   } else if (
      (cfg->ra & authz_svndb_deny_read) &&
      (cfg->ra & authz_svndb_deny_write)
   ) { /* all access denied */
      status = HTTP_FORBIDDEN;
   } else if ( /* If the user is trying to authenticate, let him. */
      apr_table_get(
         r->headers_in,
         (r->proxyreq == PROXYREQ_PROXY) ?
            "Proxy-Authorization" : "Authorization"
      )
   ) {
      status = DECLINED; /* Let auth_checker() determine access. */
   } else { /* Determine if anonymous access is allowed. */
      status = svndb_check_access(r, cfg, &repos_path, &dest_repos_path);
   }

   if (status == OK) {
      apr_table_setn(r->notes, "authz_svndb-authorized", (const char*)1);
      log_access_verdict(APLOG_MARK, r, 1, repos_path, dest_repos_path);
   }

   return status;
}

/* Authorize Anonymous Users */
static int check_user_id(request_rec *r) {
   authz_svndb_cfg *cfg = ap_get_module_config(
      r->per_dir_config,
      &authz_svndb_module
   );

   int status;
   const char *repos_path;
   const char *dest_repos_path = NULL;

   if (r->user) { /* If user is not anonymous, Do nothing. */
      status = DECLINED;
   } else if (apr_table_get(r->notes, "authz_svndb-authorized")) {
      status = OK;
   } else {
      status = svndb_check_access(r, cfg, &repos_path, &dest_repos_path);

      /* If anon access is allowed, return OK, preventing later modules
       * from issuing an HTTP_UNAUTHORIZED.  Also pass a note to our
       * auth_checker hook that access has already been checked. */
      if (status == OK) {
         apr_table_setn(r->notes, "authz_svndb-authorized", (const char*)1);
         log_access_verdict(APLOG_MARK, r, 1, repos_path, dest_repos_path);
      }
   }

   return status;
}

/* Check Non-Anonymous Access */
static int auth_checker(request_rec *r) {
   authz_svndb_cfg *cfg = ap_get_module_config(
      r->per_dir_config,
      &authz_svndb_module
   );

   int status;
   const char *repos_path;
   const char *dest_repos_path = NULL;

   /* Previous hook (check_user_id) already did all the work,
    * and, as a sanity check, r->user hasn't been set since then? */
   if (!r->user && apr_table_get(r->notes, "authz_svndb-authorized")) {
      status = OK;
   } else {
      status = svndb_check_access(r, cfg, &repos_path, &dest_repos_path);

      if (status == OK) {
         log_access_verdict(APLOG_MARK, r, 1, repos_path, dest_repos_path);
      } else if (status == DECLINED && cfg->authoritative) {
         status = HTTP_FORBIDDEN;
         log_access_verdict(APLOG_MARK, r, 0, repos_path, dest_repos_path);
         ap_note_auth_failure(r);
      }
   }

   return status;
}

/* * * Module flesh * * */

static void authz_svndb_hooks(apr_pool_t *p) {
   static const char * const mod_ssl[] = { "mod_ssl.c", NULL };

   ap_hook_access_checker(access_checker, NULL, NULL, APR_HOOK_LAST);
   /* Our check_user_id hook must be before any module which will return
    * HTTP_UNAUTHORIZED (mod_auth_basic, etc.), but after mod_ssl, to
    * give SSLOptions +FakeBasicAuth a chance to work. */
   ap_hook_check_user_id(check_user_id, mod_ssl, NULL, APR_HOOK_FIRST);
   ap_hook_auth_checker(auth_checker, NULL, NULL, APR_HOOK_FIRST);
}

module AP_MODULE_DECLARE_DATA authz_svndb_module = {
   STANDARD20_MODULE_STUFF,
   authz_svndb_init,
   authz_svndb_merge,
   NULL,
   NULL,
   authz_svndb_cmds,
   authz_svndb_hooks
};
